package au.edu.uts.aip.week1challenge;

import java.util.*;

/**
 * Transforms an iterable collection of objects into a HTML table.
 * Each object returned from the iterable is transformed into a row of the table.
 * The columns of the table are configured by calling the addColumn method.
 * @param <E> the type of each row to be formatted
 */
public class Table<E> implements HtmlGenerator<Iterable<E>> {
    
    private ArrayList<Column<E>> columns;

    /**
     * A column of the table: combines a column header (the title) with
     * a HtmlGenerator to format the cell values.
     * @param <E> the type of the row that will be formatted by the column
     */
    private static class Column<E> {
        private String title;
        private HtmlGenerator<E> generator;
        
        private Column(String title, HtmlGenerator<E> generator) {
            this.title = title;
            this.generator = generator;
        }
    }
    
    public Table() {
        columns = new ArrayList<>();
    }
    
    /**
     * Adds an additional column to the table.
     * The title is used to render the header of the table.
     * @param title the column header shown at the top of the table
     * @param generator used to produce HTML for each cell in the column
     */
    public void addColumn(String title, HtmlGenerator<E> generator) {
        columns.add(new Column<>(title, generator));
    }
    
    /**
     * Transforms the supplied iterable into an HTML table.
     * @param list the rows of the table
     * @return a HTML representation of the supplied list
     */
    @Override
    public String format(Iterable<E> list) {
        StringBuilder result = new StringBuilder();
        result.append("<table>");
        
        result.append("<tr>");
        for (Column<E> column : columns) {
            result.append("<th>");
            result.append(column.title);
            result.append("</th>");
        }
        result.append("</tr>");
        
        for (E row : list) {
            result.append("<tr>");
            for (Column<E> column : columns) {
                result.append("<td>");
                result.append(column.generator.format(row));
                result.append("</td>");
            }
            result.append("</tr>");
        }
        result.append("</table>");
        return result.toString();
    }
}
